#!/bin/bash
#
# exit 0: run success
# exit 1: need root
# exit 2: update version
# exit 3: run error
#

SOURCE="$0"

while [ -h "$SOURCE" ]; do
    DIR="$( cd -P "$( dirname "$SOURCE"  )" && pwd  )"
    SOURCE="$(readlink "$SOURCE")"
	# if $SOURCE was a relative symlink, we need to resolve it relative to the path where the symlink file was located
    [[ $SOURCE != /*  ]] && SOURCE="$DIR/$SOURCE"
done
DIR="$( cd -P "$( dirname "$SOURCE"  )" && pwd  )"

if [ $# -lt 1 ] ; then
    echo "    usage: $0 kext-path, error: too few arguments to function call, expected 1, have $#"
    exit 0
fi 

#######################################################
#                 define const variables              #
#######################################################
driver_filename=$1
running_version=''

# the driver file is in the same directory as the script file.
driver_absname="${DIR}/${driver_filename}"
echo " - kext absolute path: ${driver_absname}"

# kext file exist?
if [ ! -d ${driver_absname} ] ;  then
    echo " - kext install failed, not such file: ${driver_absname}"
    exit 1
fi

driver_version=`plutil -p ${driver_absname}/Contents/Info.plist | grep CFBundleShortVersionString | awk -F '["]' '{print $4}'`
driver_indentifier=`plutil -p ${driver_absname}/Contents/Info.plist | grep CFBundleIdentifier | awk -F '["]' '{print $4}'`

readonly driver_filename
readonly driver_indentifier
readonly driver_version

#######################################################
#         obtain the version of running driver        #
#######################################################
function is_running()
{
    # the result looks like "(1.0.0)" or None.
    running_version=`kextstat | grep ${driver_indentifier} | awk -F '[()]' '{print $2}'`
    # if the result isn't None, it means that driver is running.
    # if [ ${#running_version} != 0 ] ; then
    if [ "#${running_version}" != "#" ] ; then
        # trim the parentheses before and after.
        running_version=${running_version#*\(}
        running_version=${running_version%%\)*}

        echo " - Darwin kext is running, version: ${running_version}"
    fi
}

#######################################################
#                      load driver                    #
#######################################################
function load_driver()
{
    # check if the driver file exists.
    # note: the driver file is actually a directory.
    if [ ! -d "$1" ] ; then
        echo " - kextutil, no such file or directory: $1"
    else
        # load driver as kext.
        chown -R root:wheel $1
        chmod -R 755 $1 2> /dev/null
        kextutil -t $1
        if [ 0 == $? ] ; then
            echo " - Load the darwin kext $1 OK"
        fi 
    fi
}

#######################################################
#                          main                       #
#######################################################
function main()
{
    # check root
    if [ $EUID -ne 0 ] ; then
        echo " - startup script failed (Permission denied), are you root?"
        exit 1
    fi

    # obtain running version
    is_running $driver_indentifier
    # if the version isn't empty, it means that driver is running.
    if [ "#${running_version}" == "#" ] ; then
        # load driver
        load_driver $driver_absname

        # check if driver is running.
        is_running $driver_indentifier
        if [ "#${running_version}" == "#" ] ; then
            echo " - Load driver failed!"
            exit 3
        else
            echo " - Load driver successfully."
        fi
    fi
    
    if [ "${running_version}" == "${driver_version}" ] ; then
        echo " - Driver is running, it is the right version."
        exit 0
    else
        echo " - Please update driver: ${running_version} -> ${driver_version}"
        exit 2
    fi
}

# the entry
main
